/**
* \file: OsgCamtoGstbuf.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* Abstract base class for camtogst for OSG
*
* \component: OsgCamtoGstbuf
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#ifndef OSG_CAMTOGST_H
#define OSG_CAMTOGST_H

#include <gst/gst.h>
#include <gentexgstbuf.h>
#include <queue>

namespace osgCamtoGstbuf {

class Texture_obj;

/**
 * @brief Abstract base class for video camtogst.
 *
 * Implementations for several platforms should be able to supply an
 * texture that will be bound on the next rendering step and
 * convert such an texture to a GstBuffer.
 */
class OsgCamtoGstbuf : public generateTexGstbuf::GenTexGstbuf, public generateTexGstbuf::BufferTraits
{
public:

    pthread_t thread;
    /*For synch between Render thread and wait_client*/
    pthread_mutex_t sync_lock = PTHREAD_MUTEX_INITIALIZER;
    pthread_cond_t sync_cond = PTHREAD_COND_INITIALIZER;

    /**
     * @brief helper to create fbo
     */
    class FboTraits
    {
    public:
		FboTraits()
        :depth(0),
         stencil(0),
         sampleBuffers(0),
         samples(0){}

        // buffer depths, 0 equals off.
        unsigned int depth;
        unsigned int stencil;

        // multi sample parameters
        unsigned int sampleBuffers;
        unsigned int samples;
    };

#if GST_CHECK_VERSION(1,0,0)
    struct UserPtrData
    {
      void *osg_cam;
      void *tex;
    };
#endif

    struct rel_texobj
    {
        rel_texobj() : obj(NULL){}
        Texture_obj *  obj;
    }rel_obj;

    std::queue<rel_texobj> texobj_queue;

    /**
     * @brief create osgcamtogstbuf object
     * @params BufferTraits ,information for Texture creation
     *         FboTraits ,information for framebuffer creation
     */
    OsgCamtoGstbuf(const BufferTraits &txoTraits, const FboTraits &fboTraits);

    /**
     * @brief set the width and height of the buffer
     * @param width new width,height new height
     */
    void setBuffersize(uint width, uint height);

    /**
     * @brief get the width and height of the buffer
     * @return width new width,height new height
     */
    void getBuffersize(uint *width, uint *height);

    /**
     * @brief set Gst buffer callback
     * @param object and callback function
     */
	void setBufferCallback(void *obj ,void (*cb)(void *,void *));

    /**
     * @brief Get the camera associated and setup for the camtogst
     * @return instance of the camera
     */
    virtual osg::Camera *getOrCreateCamera();

    /**
     * @brief Attach the camtogst to an existing osg::Camera instance
     * @param camera
     *
     * After this call, getOrCreateCamera() will return the attached camera.
     * If a camera was previously attached to the camtogst, it will be detached
     * first. This can only be called if the camtogst is not running.
     */
    void attachToCamera(osg::Camera *camera);

    /**
     * @brief Detach the camtogst from an existing osg::Camera instance
     * @param camera
     *
     * This can only be called if the camtogst is not running.
     */
    void detachFromCamera(osg::Camera *camera);

    const FboTraits& getFramebufferTraits() const;

    /**
     * @brief push a frame into the gstreamer pipeline
     * @param txo
     */
    void push( Texture_obj *txo);

#if GST_CHECK_VERSION(1,0,0)
    static void gst_buffer_final (struct UserPtrData *data);
#endif

    virtual ~OsgCamtoGstbuf();

private:
    class OsgCamtoGstbufPrivate;
    std::auto_ptr<OsgCamtoGstbufPrivate> _p;
    OsgCamtoGstbuf *cam_to_gst_obj = NULL;
    generateTexGstbuf::GenTexGstbuf *gen_tex_gstbuf = NULL;

    void (*GstBuffer_callback)(void *, void *);
    void *gst_buffer_obj;

    void setFramebufferTraits(const FboTraits &traits);
};
} // namespace osgCamtoGstbuf
#endif // OSG_CAMTOGST_H
